/***************************************************************************
 *   Copyright (C) 2007 by Faubet Pierre   *
 *   pierre.faubet@e.ujf-grenoble.fr   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef ENVIRONMENTAL_FACTORS_H
#define ENVIRONMENTAL_FACTORS_H

#include <string>

using namespace std;

/**
* This class implements data structure for biotic/abiotic factors
* @author Faubet Pierre <pierre.faubet@e.ujf-grenoble.fr>
*/
class environmental_factors
  {
  private:
    double ***G; /**< Values for the biotic/abiotic factors */
    bool with_interaction; /**< Whether we consider or not first-order interactions between factors */
    int nb_factors; /**< Number of environmental factors considered */
    int size; /**< Size of G, depends on the consideration of interactions */
    int nb_model;
    int nb_pop; /**< Number of populations */

    string inputfilename;

    /**
     * Create arrays to store data
     */
    void createarrays();

    /**
     * Normalize factors with mean 0 and variance 1
     */
    void normalize();

    /**
     * Add interactions if considered
     */
    void addinteractions();

  public:
    /**
    * Constructor: read environmental factor values
    * @param filename Name of the inputfile that contains environmental factor values
    * @param interaction true if we consider first-order interactions, false otherwise
    */
    environmental_factors(const string& filename,bool interaction);

    /**
     * Write environmental data into an output file
     * @param outfil Name of the outputfile
     */
    void write(const string& outfil);

    /**
    * Check that a given model is valid
    * @param M model code
    * @return true if model M is valid, false otherwise
    */
    bool is_a_valid_model(int M);

    ~environmental_factors();

    string get_inputfilename() const
      {
        return inputfilename;
      }

    double get_G(int r,int q,int l) const
      {
        return G[r][q][l];
      }

    int get_nb_factors() const
      {
        return nb_factors;
      }
    int get_size() const
      {
        return size;
      }

    bool get_with_interaction() const
      {
        return with_interaction;
      }

    int get_nb_model() const
      {
        return nb_model;
      }
  };


#endif
