/***************************************************************************
*   Copyright (C) 2007 by Faubet Pierre   *
*   pierre.faubet@e.ujf-grenoble.fr   *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/

#include <QtGui>
#include <QFileInfo>
#include <QFileDialog>

#include "wizard.h"

Wizard::Wizard(QWidget *parent)
    : QWizard(parent)
{
  setPage(Page_Intro, new TitlePage);
  setPage(Page_Input, new InputPage);
  setPage(Page_Settings, new SettingsPage);
  setPage(Page_Output, new OutputPage);
  setPage(Page_StartRuns, new FinishPage);
  setPage(Page_OpenProject, new OpenPage);

  setStartId(Page_Intro);

#ifndef Q_WS_MAC
  setWizardStyle(ModernStyle);
#endif
  setOption(HaveHelpButton, true);

  connect(this, SIGNAL(helpRequested()), this, SLOT(showHelp()));


  setWindowTitle(tr("BIMr Wizard"));
}

void Wizard::showHelp()
{
  static QString lastHelpMessage;

  QString message;

  switch (currentId())
    {
    case Page_Intro:
      message = tr("You have to choose between two tasks "
                   "Run or load BIMr analyses");
      break;
    case Page_Input:
      message = tr("Make sure to provide valid filenames. "
		  "Corresponding files must exits");
      break;
    case Page_Settings:
      message = tr("MCMC run, prior and proposal settings. "
		  "Give appropriate values");
      break;
    case Page_Output:
      message = tr("Monitored parameters. "
		  "Choose the ones of interest");
      break;
    case Page_StartRuns:
      message = tr("Start analysis? "
		  "Or change/check your settings");
      break;
    case Page_OpenProject:
      message = tr("Load saved work. "
		  "Open BIMr project file");
      break;
    default:
      message = tr("This help is likely not to be of any help.");
    }

  if (lastHelpMessage == message)
    message = tr("Sorry, I already gave what help I could. "
                 "Maybe you should try reading BIMr manual?");

  QMessageBox::information(this, tr("BIMr Wizard Help"), message);

  lastHelpMessage = message;
}

TitlePage::TitlePage(QWidget *parent)
    : QWizardPage(parent)
{
  QVBoxLayout *vboxLayout = new QVBoxLayout;
  vboxLayout->setSpacing(6);
  vboxLayout->setMargin(9);
  vboxLayout->setObjectName(QString::fromUtf8("vboxLayout"));

  setTitle(tr("BIMr version 1.0"));
  setSubTitle(tr("This wizard will help you to run MCMC analysis"));

  groupBox = new QGroupBox();
  groupBox->setObjectName(QString::fromUtf8("groupBox"));
  QVBoxLayout *vboxLayout1 = new QVBoxLayout(groupBox);
  vboxLayout1->setSpacing(6);
  vboxLayout1->setMargin(9);
  vboxLayout1->setObjectName(QString::fromUtf8("vboxLayout1"));

  StartradioButton = new QRadioButton(groupBox);
  StartradioButton->setObjectName(QString::fromUtf8("StartradioButton"));
  StartradioButton->setChecked(true);
  vboxLayout1->addWidget(StartradioButton);

  loadradioButton = new QRadioButton(groupBox);
  loadradioButton->setObjectName(QString::fromUtf8("loadradioButton"));
  vboxLayout1->addWidget(loadradioButton);

  groupBox->setTitle(QApplication::translate("Dialog", "You want to", 0, QApplication::UnicodeUTF8));
  StartradioButton->setText(QApplication::translate("Dialog", "perform new analyses", 0, QApplication::UnicodeUTF8));
  loadradioButton->setText(QApplication::translate("Dialog", "load previous results", 0, QApplication::UnicodeUTF8));
  vboxLayout->addWidget(groupBox);
  setLayout(vboxLayout);
}

int TitlePage::nextId() const
  {
    if (StartradioButton->isChecked())
      return Wizard::Page_Input;
    else
      return Wizard::Page_OpenProject;
  }

InputPage::InputPage(QWidget *parent)
    : QWizardPage(parent)
{
  QVBoxLayout *vboxLayout = new QVBoxLayout;
  vboxLayout->setSpacing(6);
  vboxLayout->setMargin(9);
  vboxLayout->setObjectName(QString::fromUtf8("vboxLayout"));

  setTitle(tr("Data input files"));

  groupBox = new QGroupBox();
  groupBox->setObjectName(QString::fromUtf8("groupBox"));
  QVBoxLayout *vboxLayout1 = new QVBoxLayout(groupBox);
  vboxLayout1->setSpacing(6);
  vboxLayout1->setMargin(9);
  vboxLayout1->setObjectName(QString::fromUtf8("vboxLayout1"));
  QHBoxLayout *hboxLayout = new QHBoxLayout();
  hboxLayout->setSpacing(6);
  hboxLayout->setMargin(0);
  hboxLayout->setObjectName(QString::fromUtf8("hboxLayout"));

  /* Genetic data */
  Inputgenlabel = new QLabel(groupBox);
  Inputgenlabel->setObjectName(QString::fromUtf8("Inputgenlabel"));
  hboxLayout->addWidget(Inputgenlabel);
  InputgenlineEdit = new QLineEdit(groupBox);
  InputgenlineEdit->setObjectName(QString::fromUtf8("InputgenlineEdit"));
  hboxLayout->addWidget(InputgenlineEdit);
  InputgentoolButton = new QToolButton(groupBox);
  InputgentoolButton->setObjectName(QString::fromUtf8("InputgentoolButton"));
  hboxLayout->addWidget(InputgentoolButton);

  vboxLayout1->addLayout(hboxLayout);
  vboxLayout->addWidget(groupBox);

  FactorsgroupBox = new QGroupBox;
  FactorsgroupBox->setObjectName(QString::fromUtf8("FactorsgroupBox"));
  FactorsgroupBox->setCheckable(true);
  FactorsgroupBox->setChecked(false);
  QVBoxLayout *vboxLayout2 = new QVBoxLayout(FactorsgroupBox);
  vboxLayout2->setSpacing(6);
  vboxLayout2->setMargin(9);
  vboxLayout2->setObjectName(QString::fromUtf8("vboxLayout2"));
  QHBoxLayout *hboxLayout1 = new QHBoxLayout();
  hboxLayout1->setSpacing(6);
  hboxLayout1->setMargin(0);
  hboxLayout1->setObjectName(QString::fromUtf8("hboxLayout1"));

  /* Environemental factors */
  Inputfactlabel = new QLabel(FactorsgroupBox);
  Inputfactlabel->setObjectName(QString::fromUtf8("Inputfactlabel"));
  hboxLayout1->addWidget(Inputfactlabel);
  InputfactlineEdit = new QLineEdit(FactorsgroupBox);
  InputfactlineEdit->setObjectName(QString::fromUtf8("InputfactlineEdit"));
  hboxLayout1->addWidget(InputfactlineEdit);
  InputfacttoolButton = new QToolButton(FactorsgroupBox);
  InputfacttoolButton->setObjectName(QString::fromUtf8("InputfacttoolButton"));
  hboxLayout1->addWidget(InputfacttoolButton);
  vboxLayout2->addLayout(hboxLayout1);
  WithinteractioncheckBox = new QCheckBox(FactorsgroupBox);
  WithinteractioncheckBox->setObjectName(QString::fromUtf8("WithinteractioncheckBox"));
  WithinteractioncheckBox->setChecked(true);
  vboxLayout2->addWidget(WithinteractioncheckBox);

  groupBox->setTitle(QApplication::translate("Dialog", "Multilocus genotype data", 0, QApplication::UnicodeUTF8));
  Inputgenlabel->setText(QApplication::translate("Dialog", "Input file: ", 0, QApplication::UnicodeUTF8));
  InputgentoolButton->setText(QApplication::translate("Dialog", "...", 0, QApplication::UnicodeUTF8));
  FactorsgroupBox->setTitle(QApplication::translate("Dialog", "Environmental factors", 0, QApplication::UnicodeUTF8));
  Inputfactlabel->setText(QApplication::translate("Dialog", "Input file: ", 0, QApplication::UnicodeUTF8));
  InputfacttoolButton->setText(QApplication::translate("Dialog", "...", 0, QApplication::UnicodeUTF8));
  WithinteractioncheckBox->setText(QApplication::translate("Dialog", "Add first order interactions", 0, QApplication::UnicodeUTF8));

  InputgenlineEdit->setText(QDir::homePath());
  InputfactlineEdit->setText(QDir::homePath());

  completer = new QCompleter(this);
  completer->setModel(new QDirModel(completer));
  InputgenlineEdit->setCompleter(completer);
  InputfactlineEdit->setCompleter(completer);
  vboxLayout->addWidget(FactorsgroupBox);
  setLayout(vboxLayout);

  connect(InputgenlineEdit, SIGNAL(textChanged(QString)),this, SIGNAL(completeChanged()));
  connect(InputfactlineEdit, SIGNAL(textChanged(QString)),this, SIGNAL(completeChanged()));
  connect(FactorsgroupBox, SIGNAL(toggled(bool)),this, SIGNAL(completeChanged()));
  connect(InputgentoolButton,SIGNAL(clicked(bool)),this,SLOT(getFilename()));
  connect(InputfacttoolButton,SIGNAL(clicked(bool)),this,SLOT(getFilename()));

  registerField("gen_filename",InputgenlineEdit);
  registerField("env_filename",InputfactlineEdit);
  registerField("NGData",FactorsgroupBox,"checked");
  registerField("inter",WithinteractioncheckBox,"checked");
}

bool InputPage::isComplete() const
  {
    QFileInfo info1(InputgenlineEdit->text());

    if (FactorsgroupBox->isChecked())
      {
        QFileInfo info2(InputfactlineEdit->text());
        return info1.isFile() && info2.isFile();
      }
    return info1.isFile();
  }

int InputPage::nextId() const
  {
    return Wizard::Page_Settings;
  }

void InputPage::getFilename()
{
  QString caption;
  QString filter;
  QLineEdit *filelineEdit = NULL;

  /* Non genetic data */
  if (sender()->objectName() == InputfacttoolButton->objectName())
    {
      caption = tr("Open environmental factor input file");
      filter = tr("Text Files (*.txt);;All Files (*)");
      filelineEdit = InputfactlineEdit;
    }
  /* Genetic data */
  else if (sender()->objectName() == InputgentoolButton->objectName())
    {
      caption = tr("Open multilocus genotype input file");
      filter = tr("Gen Files (*.gen);;All Files (*)");
      filelineEdit = InputgenlineEdit;
    }
  else
    {
      qDebug() << "Error";
    }
  QString fn = QFileDialog::getOpenFileName(this,caption,QDir::homePath(),filter);
  if (!fn.isEmpty())
    filelineEdit->setText(fn);
}

SettingsPage::SettingsPage(QWidget *parent)
    : QWizardPage(parent)
{
  QVBoxLayout *vboxLayout = new QVBoxLayout;
  vboxLayout->setSpacing(6);
  vboxLayout->setMargin(9);
  vboxLayout->setObjectName(QString::fromUtf8("vboxLayout"));

  setTitle(tr("Settings"));

  QHBoxLayout *hboxLayout = new QHBoxLayout();
  hboxLayout->setSpacing(6);
  hboxLayout->setMargin(0);
  hboxLayout->setObjectName(QString::fromUtf8("hboxLayout"));
  QVBoxLayout *vboxLayout1 = new QVBoxLayout();
  vboxLayout1->setSpacing(6);
  vboxLayout1->setMargin(0);
  vboxLayout1->setObjectName(QString::fromUtf8("vboxLayout1"));
  SettingsgroupBox = new QGroupBox();
  SettingsgroupBox->setObjectName(QString::fromUtf8("SettingsgroupBox"));
  QGridLayout *gridLayout = new QGridLayout(SettingsgroupBox);
  gridLayout->setSpacing(6);
  gridLayout->setMargin(9);
  gridLayout->setObjectName(QString::fromUtf8("gridLayout"));

  /* Burnin */
  Burninlabel = new QLabel(SettingsgroupBox);
  Burninlabel->setObjectName(QString::fromUtf8("Burninlabel"));
  gridLayout->addWidget(Burninlabel, 0, 0, 1, 1);
  BurninlineEdit = new QLineEdit(SettingsgroupBox);
  BurninlineEdit->setObjectName(QString::fromUtf8("BurninlineEdit"));
  BurninlineEdit->setAlignment(Qt::AlignRight);
  gridLayout->addWidget(BurninlineEdit, 0, 1, 1, 2);

  /* Sample size */
  SampleSizelabel = new QLabel(SettingsgroupBox);
  SampleSizelabel->setObjectName(QString::fromUtf8("SampleSizelabel"));
  gridLayout->addWidget(SampleSizelabel, 1, 0, 1, 1);
  vboxLayout1->addWidget(SettingsgroupBox);
  SampleSizelineEdit = new QLineEdit(SettingsgroupBox);
  SampleSizelineEdit->setObjectName(QString::fromUtf8("SampleSizelineEdit"));
  SampleSizelineEdit->setAlignment(Qt::AlignRight);
  gridLayout->addWidget(SampleSizelineEdit, 1, 1, 1, 2);

  /* Thining */
  Thininglabel = new QLabel(SettingsgroupBox);
  Thininglabel->setObjectName(QString::fromUtf8("Thininglabel"));
  gridLayout->addWidget(Thininglabel, 2, 0, 1, 1);
  ThininglineEdit = new QLineEdit(SettingsgroupBox);
  ThininglineEdit->setObjectName(QString::fromUtf8("ThininglineEdit"));
  ThininglineEdit->setAlignment(Qt::AlignRight);
  gridLayout->addWidget(ThininglineEdit, 2, 1, 1, 2);

  /* Number of replicates */
  NbReplabel = new QLabel(SettingsgroupBox);
  NbReplabel->setObjectName(QString::fromUtf8("NbReplabel"));
  gridLayout->addWidget(NbReplabel, 3, 0, 1, 2);
  NbRepspinBox = new QSpinBox(SettingsgroupBox);
  NbRepspinBox->setObjectName(QString::fromUtf8("NbRepspinBox"));
  NbRepspinBox->setAlignment(Qt::AlignRight);
  NbRepspinBox->setMaximum(50);
  NbRepspinBox->setMinimum(1);
  NbRepspinBox->setValue(3);
  gridLayout->addWidget(NbRepspinBox, 3, 2, 1, 1);

  AdancedgroupBox = new QGroupBox();
  AdancedgroupBox->setObjectName(QString::fromUtf8("AdancedgroupBox"));
  QVBoxLayout *vboxLayout2 = new QVBoxLayout(AdancedgroupBox);
  vboxLayout2->setSpacing(6);
  vboxLayout2->setMargin(9);
  vboxLayout2->setObjectName(QString::fromUtf8("vboxLayout2"));

  /* Reversible jump MCMC */
  useRJMCMCcheckBox = new QCheckBox(AdancedgroupBox);
  useRJMCMCcheckBox->setObjectName(QString::fromUtf8("useRJMCMCcheckBox"));
  useRJMCMCcheckBox->setChecked(true);
  vboxLayout2->addWidget(useRJMCMCcheckBox);

  /* Correlated allele frequencies */
  useFmodelcheckBox = new QCheckBox(AdancedgroupBox);
  useFmodelcheckBox->setObjectName(QString::fromUtf8("useFmodelcheckBox"));
  useFmodelcheckBox->setChecked(true);
  vboxLayout2->addWidget(useFmodelcheckBox);

  /* Pilot runs */
  PilotrungroupBox = new QGroupBox(AdancedgroupBox);
  PilotrungroupBox->setObjectName(QString::fromUtf8("PilotrungroupBox"));
  PilotrungroupBox->setCheckable(true);
  PilotrungroupBox->setChecked(true);
  QGridLayout *gridLayout1 = new QGridLayout(PilotrungroupBox);
  gridLayout1->setSpacing(6);
  gridLayout1->setMargin(9);
  gridLayout1->setObjectName(QString::fromUtf8("gridLayout1"));

  Nbpilotlabel = new QLabel(PilotrungroupBox);
  Nbpilotlabel->setObjectName(QString::fromUtf8("Nbpilotlabel"));
  gridLayout1->addWidget(Nbpilotlabel, 0, 0, 1, 1);

  NbpilotspinBox = new QSpinBox(PilotrungroupBox);
  NbpilotspinBox->setObjectName(QString::fromUtf8("NbpilotspinBox"));
  NbpilotspinBox->setAlignment(Qt::AlignRight);
  NbpilotspinBox->setValue(20);
  gridLayout1->addWidget(NbpilotspinBox, 0, 1, 1, 1);

  PilotlengthspinBox = new QSpinBox(PilotrungroupBox);
  PilotlengthspinBox->setObjectName(QString::fromUtf8("PilotlengthspinBox"));
  PilotlengthspinBox->setAlignment(Qt::AlignRight);
  PilotlengthspinBox->setMaximum(100000);
  PilotlengthspinBox->setSingleStep(500);
  PilotlengthspinBox->setValue(1000);
  gridLayout1->addWidget(PilotlengthspinBox, 1, 1, 1, 1);

  Pilotlengthlabel = new QLabel(PilotrungroupBox);
  Pilotlengthlabel->setObjectName(QString::fromUtf8("Pilotlengthlabel"));
  gridLayout1->addWidget(Pilotlengthlabel, 1, 0, 1, 1);

  vboxLayout2->addWidget(PilotrungroupBox);

  vboxLayout1->addWidget(AdancedgroupBox);

  hboxLayout->addLayout(vboxLayout1);

  QHBoxLayout *hboxLayout1 = new QHBoxLayout();
  hboxLayout1->setSpacing(6);
  hboxLayout1->setMargin(0);
  hboxLayout1->setObjectName(QString::fromUtf8("hboxLayout1"));

  PriorgroupBox = new QGroupBox();
  PriorgroupBox->setObjectName(QString::fromUtf8("PriorgroupBox"));
  QVBoxLayout *vboxLayout3 = new QVBoxLayout(PriorgroupBox);
  vboxLayout3->setSpacing(6);
  vboxLayout3->setMargin(9);
  vboxLayout3->setObjectName(QString::fromUtf8("vboxLayout3"));
  QGridLayout *gridLayout2 = new QGridLayout();
  gridLayout2->setSpacing(6);
  gridLayout2->setMargin(0);
  gridLayout2->setObjectName(QString::fromUtf8("gridLayout2"));

  s2alphalabel = new QLabel(PriorgroupBox);
  s2alphalabel->setObjectName(QString::fromUtf8("s2alphalabel"));
  gridLayout2->addWidget(s2alphalabel, 0, 0, 1, 1);
  s2_alphadoubleSpinBox = new QDoubleSpinBox(PriorgroupBox);
  s2_alphadoubleSpinBox->setObjectName(QString::fromUtf8("s2_alphadoubleSpinBox"));
  s2_alphadoubleSpinBox->setAlignment(Qt::AlignRight);
  s2_alphadoubleSpinBox->setSingleStep(0.5);
  s2_alphadoubleSpinBox->setValue(4);
  gridLayout2->addWidget(s2_alphadoubleSpinBox, 0, 1, 1, 1);


  a_taulabel = new QLabel(PriorgroupBox);
  a_taulabel->setObjectName(QString::fromUtf8("a_taulabel"));
  gridLayout2->addWidget(a_taulabel, 1, 0, 1, 1);
  a_taudoubleSpinBox = new QDoubleSpinBox(PriorgroupBox);
  a_taudoubleSpinBox->setObjectName(QString::fromUtf8("a_taudoubleSpinBox"));
  a_taudoubleSpinBox->setAlignment(Qt::AlignRight);
  a_taudoubleSpinBox->setSingleStep(0.5);
  a_taudoubleSpinBox->setValue(1);
  gridLayout2->addWidget(a_taudoubleSpinBox, 1, 1, 1, 1);

  b_taulabel = new QLabel(PriorgroupBox);
  b_taulabel->setObjectName(QString::fromUtf8("b_taulabel"));
  gridLayout2->addWidget(b_taulabel, 2, 0, 1, 1);
  b_taudoubleSpinBox = new QDoubleSpinBox(PriorgroupBox);
  b_taudoubleSpinBox->setObjectName(QString::fromUtf8("b_taudoubleSpinBox"));
  b_taudoubleSpinBox->setAlignment(Qt::AlignRight);
  b_taudoubleSpinBox->setSingleStep(0.5);
  b_taudoubleSpinBox->setValue(1);
  gridLayout2->addWidget(b_taudoubleSpinBox, 2, 1, 1, 1);

  vboxLayout3->addLayout(gridLayout2);

  line_2 = new QFrame(PriorgroupBox);
  line_2->setObjectName(QString::fromUtf8("line_2"));
  line_2->setFrameShape(QFrame::HLine);

  vboxLayout3->addWidget(line_2);

  QHBoxLayout *hboxLayout2 = new QHBoxLayout();
  hboxLayout2->setSpacing(6);
  hboxLayout2->setMargin(0);
  hboxLayout2->setObjectName(QString::fromUtf8("hboxLayout2"));

  psilabel = new QLabel(PriorgroupBox);
  psilabel->setObjectName(QString::fromUtf8("psilabel"));
  hboxLayout2->addWidget(psilabel);
  psidoubleSpinBox = new QDoubleSpinBox(PriorgroupBox);
  psidoubleSpinBox->setObjectName(QString::fromUtf8("psidoubleSpinBox"));
  psidoubleSpinBox->setAlignment(Qt::AlignRight);
  psidoubleSpinBox->setSingleStep(0.5);
  psidoubleSpinBox->setValue(1);
  hboxLayout2->addWidget(psidoubleSpinBox);

  vboxLayout3->addLayout(hboxLayout2);

  line_3 = new QFrame(PriorgroupBox);
  line_3->setObjectName(QString::fromUtf8("line_3"));
  line_3->setFrameShape(QFrame::HLine);
  vboxLayout3->addWidget(line_3);

  QGridLayout *gridLayout3 = new QGridLayout();
  gridLayout3->setSpacing(6);
  gridLayout3->setMargin(0);
  gridLayout3->setObjectName(QString::fromUtf8("gridLayout3"));

  lambdalabel = new QLabel(PriorgroupBox);
  lambdalabel->setObjectName(QString::fromUtf8("lambdalabel"));
  gridLayout3->addWidget(lambdalabel, 0, 0, 1, 1);
  lambdadoubleSpinBox = new QDoubleSpinBox(PriorgroupBox);
  lambdadoubleSpinBox->setObjectName(QString::fromUtf8("lambdadoubleSpinBox"));
  lambdadoubleSpinBox->setAlignment(Qt::AlignRight);
  lambdadoubleSpinBox->setSingleStep(0.5);
  lambdadoubleSpinBox->setValue(1);
  gridLayout3->addWidget(lambdadoubleSpinBox, 0, 1, 1, 1);

  omegalabel = new QLabel(PriorgroupBox);
  omegalabel->setObjectName(QString::fromUtf8("omegalabel"));
  gridLayout3->addWidget(omegalabel, 1, 0, 1, 1);
  omegadoubleSpinBox = new QDoubleSpinBox(PriorgroupBox);
  omegadoubleSpinBox->setObjectName(QString::fromUtf8("omegadoubleSpinBox"));
  omegadoubleSpinBox->setAlignment(Qt::AlignRight);
  omegadoubleSpinBox->setSingleStep(0.5);
  omegadoubleSpinBox->setValue(1);
  gridLayout3->addWidget(omegadoubleSpinBox, 1, 1, 1, 1);

  xilabel = new QLabel(PriorgroupBox);
  xilabel->setObjectName(QString::fromUtf8("xilabel"));
  gridLayout3->addWidget(xilabel, 2, 0, 1, 1);
  xidoubleSpinBox = new QDoubleSpinBox(PriorgroupBox);
  xidoubleSpinBox->setObjectName(QString::fromUtf8("xidoubleSpinBox"));
  xidoubleSpinBox->setAlignment(Qt::AlignRight);
  xidoubleSpinBox->setSingleStep(0.5);
  xidoubleSpinBox->setValue(1);
  gridLayout3->addWidget(xidoubleSpinBox, 2, 1, 1, 1);

  vboxLayout3->addLayout(gridLayout3);

  hboxLayout1->addWidget(PriorgroupBox);

  ProposalgroupBox = new QGroupBox();
  ProposalgroupBox->setObjectName(QString::fromUtf8("ProposalgroupBox"));
  QGridLayout *gridLayout4 = new QGridLayout(ProposalgroupBox);
  gridLayout4->setSpacing(6);
  gridLayout4->setMargin(9);
  gridLayout4->setObjectName(QString::fromUtf8("gridLayout4"));

  e_mlabel = new QLabel(ProposalgroupBox);
  e_mlabel->setObjectName(QString::fromUtf8("e_mlabel"));
  gridLayout4->addWidget(e_mlabel, 0, 0, 1, 1);
  e_mdoubleSpinBox = new QDoubleSpinBox(ProposalgroupBox);
  e_mdoubleSpinBox->setObjectName(QString::fromUtf8("e_mdoubleSpinBox"));
  e_mdoubleSpinBox->setAlignment(Qt::AlignRight);
  e_mdoubleSpinBox->setMaximum(1);
  e_mdoubleSpinBox->setMinimum(0.01);
  e_mdoubleSpinBox->setSingleStep(0.01);
  e_mdoubleSpinBox->setValue(0.1);
  gridLayout4->addWidget(e_mdoubleSpinBox, 0, 1, 1, 1);

  e_nmlabel = new QLabel(ProposalgroupBox);
  e_nmlabel->setObjectName(QString::fromUtf8("e_nmlabel"));
  gridLayout4->addWidget(e_nmlabel, 1, 0, 1, 1);
  e_nmdoubleSpinBox = new QDoubleSpinBox(ProposalgroupBox);
  e_nmdoubleSpinBox->setObjectName(QString::fromUtf8("e_mdoubleSpinBox"));
  e_nmdoubleSpinBox->setAlignment(Qt::AlignRight);
  e_nmdoubleSpinBox->setMaximum(1);
  e_nmdoubleSpinBox->setMinimum(0.01);
  e_nmdoubleSpinBox->setSingleStep(0.01);
  e_nmdoubleSpinBox->setValue(0.1);
  gridLayout4->addWidget(e_nmdoubleSpinBox, 1, 1, 1, 1);

  e_Flabel = new QLabel(ProposalgroupBox);
  e_Flabel->setObjectName(QString::fromUtf8("e_Flabel"));
  gridLayout4->addWidget(e_Flabel, 2, 0, 1, 1);
  e_FdoubleSpinBox = new QDoubleSpinBox(ProposalgroupBox);
  e_FdoubleSpinBox->setObjectName(QString::fromUtf8("e_FdoubleSpinBox"));
  e_FdoubleSpinBox->setAlignment(Qt::AlignRight);
  e_FdoubleSpinBox->setMaximum(1);
  e_FdoubleSpinBox->setMinimum(0.01);
  e_FdoubleSpinBox->setSingleStep(0.01);
  e_FdoubleSpinBox->setValue(0.1);
  gridLayout4->addWidget(e_FdoubleSpinBox, 2, 1, 1, 1);

  e_plabel = new QLabel(ProposalgroupBox);
  e_plabel->setObjectName(QString::fromUtf8("e_plabel"));
  gridLayout4->addWidget(e_plabel, 3, 0, 1, 1);
  e_pdoubleSpinBox = new QDoubleSpinBox(ProposalgroupBox);
  e_pdoubleSpinBox->setObjectName(QString::fromUtf8("e_pdoubleSpinBox"));
  e_pdoubleSpinBox->setAlignment(Qt::AlignRight);
  e_pdoubleSpinBox->setMaximum(1);
  e_pdoubleSpinBox->setMinimum(0.01);
  e_pdoubleSpinBox->setSingleStep(0.01);
  e_pdoubleSpinBox->setValue(0.1);
  gridLayout4->addWidget(e_pdoubleSpinBox, 3, 1, 1, 1);

  s2thetalabel = new QLabel(ProposalgroupBox);
  s2thetalabel->setObjectName(QString::fromUtf8("s2thetalabel"));
  gridLayout4->addWidget(s2thetalabel, 4, 0, 1, 1);
  s2thetadoubleSpinBox = new QDoubleSpinBox(ProposalgroupBox);
  s2thetadoubleSpinBox->setObjectName(QString::fromUtf8("s2thetadoubleSpinBox"));
  s2thetadoubleSpinBox->setAlignment(Qt::AlignRight);
  s2thetadoubleSpinBox->setSingleStep(0.5);
  s2thetadoubleSpinBox->setValue(5);
  gridLayout4->addWidget(s2thetadoubleSpinBox, 4, 1, 1, 1);

  s2_psilabel = new QLabel(ProposalgroupBox);
  s2_psilabel->setObjectName(QString::fromUtf8("s2_psilabel"));
  gridLayout4->addWidget(s2_psilabel, 5, 0, 1, 1);
  s2psidoubleSpinBox = new QDoubleSpinBox(ProposalgroupBox);
  s2psidoubleSpinBox->setObjectName(QString::fromUtf8("s2psidoubleSpinBox"));
  s2psidoubleSpinBox->setAlignment(Qt::AlignRight);
  s2psidoubleSpinBox->setSingleStep(0.5);
  s2psidoubleSpinBox->setValue(5);
  gridLayout4->addWidget(s2psidoubleSpinBox, 5, 1, 1, 1);

  hboxLayout1->addWidget(ProposalgroupBox);

  hboxLayout->addLayout(hboxLayout1);

  SettingsgroupBox->setTitle(QApplication::translate("Dialog", "Iterations", 0, QApplication::UnicodeUTF8));
  Burninlabel->setText(QApplication::translate("Dialog", "Burnin", 0, QApplication::UnicodeUTF8));
  SampleSizelineEdit->setText(QApplication::translate("Dialog", "10000", 0, QApplication::UnicodeUTF8));
  BurninlineEdit->setText(QApplication::translate("Dialog", "10000", 0, QApplication::UnicodeUTF8));
  NbReplabel->setText(QApplication::translate("Dialog", "Number of replicates", 0, QApplication::UnicodeUTF8));
  Thininglabel->setText(QApplication::translate("Dialog", "Thining interval", 0, QApplication::UnicodeUTF8));
  ThininglineEdit->setText(QApplication::translate("Dialog", "50", 0, QApplication::UnicodeUTF8));
  SampleSizelabel->setText(QApplication::translate("Dialog", "Sample size", 0, QApplication::UnicodeUTF8));
  AdancedgroupBox->setTitle(QApplication::translate("Dialog", "Advanced features", 0, QApplication::UnicodeUTF8));
  useFmodelcheckBox->setText(QApplication::translate("Dialog", "F-model", 0, QApplication::UnicodeUTF8));
  useRJMCMCcheckBox->setText(QApplication::translate("Dialog", "Reversible Jump MCMC", 0, QApplication::UnicodeUTF8));
  PilotrungroupBox->setTitle(QApplication::translate("Dialog", "Pilot runs", 0, QApplication::UnicodeUTF8));
  Nbpilotlabel->setText(QApplication::translate("Dialog", "Number", 0, QApplication::UnicodeUTF8));
  Pilotlengthlabel->setText(QApplication::translate("Dialog", "Length", 0, QApplication::UnicodeUTF8));
  PriorgroupBox->setTitle(QApplication::translate("Dialog", "Prior", 0, QApplication::UnicodeUTF8));
  a_taulabel->setText(QApplication::translate("Dialog", "a_tau", 0, QApplication::UnicodeUTF8));
  s2alphalabel->setText(QApplication::translate("Dialog", "s2_alpha", 0, QApplication::UnicodeUTF8));
  b_taulabel->setText(QApplication::translate("Dialog", "b_tau", 0, QApplication::UnicodeUTF8));
  psilabel->setText(QApplication::translate("Dialog", "psi", 0, QApplication::UnicodeUTF8));
  omegalabel->setText(QApplication::translate("Dialog", "omega", 0, QApplication::UnicodeUTF8));
  lambdalabel->setText(QApplication::translate("Dialog", "lambda", 0, QApplication::UnicodeUTF8));
  xilabel->setText(QApplication::translate("Dialog", "xi", 0, QApplication::UnicodeUTF8));
  ProposalgroupBox->setTitle(QApplication::translate("Dialog", "Incremental values", 0, QApplication::UnicodeUTF8));
  s2_psilabel->setText(QApplication::translate("Dialog", "s2_psi", 0, QApplication::UnicodeUTF8));
  e_Flabel->setText(QApplication::translate("Dialog", "e_F", 0, QApplication::UnicodeUTF8));
  s2thetalabel->setText(QApplication::translate("Dialog", "s2_theta", 0, QApplication::UnicodeUTF8));
  e_plabel->setText(QApplication::translate("Dialog", "e_p", 0, QApplication::UnicodeUTF8));
  e_mlabel->setText(QApplication::translate("Dialog", "e_m", 0, QApplication::UnicodeUTF8));
  e_nmlabel->setText(QApplication::translate("Dialog", "e_nm", 0, QApplication::UnicodeUTF8));

  vboxLayout->addLayout(hboxLayout);
  setLayout(vboxLayout);

  connect(useFmodelcheckBox, SIGNAL(stateChanged(int)),this,SLOT(setEnabled()));

  registerField("burnin",BurninlineEdit);
  registerField("smpSize",SampleSizelineEdit);
  registerField("thining",ThininglineEdit);
  registerField("nrep",NbRepspinBox);

  registerField("usepilot",PilotrungroupBox,"checked");
  registerField("npilot",NbpilotspinBox);
  registerField("pilotlength",PilotlengthspinBox);

  registerField("RJ",useRJMCMCcheckBox,"checked");

  registerField("Fst",useFmodelcheckBox,"checked");

  registerField("s2_alpha",s2_alphadoubleSpinBox,"value");
  registerField("a_tau",a_taudoubleSpinBox,"value");
  registerField("b_tau",b_taudoubleSpinBox,"value");

  registerField("psi",psidoubleSpinBox,"value");

  registerField("lambda",lambdadoubleSpinBox,"value");
  registerField("omega",omegadoubleSpinBox,"value");
  registerField("xi",xidoubleSpinBox,"value");

  registerField("e_m",e_mdoubleSpinBox,"value");
  registerField("e_nm",e_nmdoubleSpinBox,"value");
  registerField("e_F",e_FdoubleSpinBox,"value");
  registerField("e_p",e_pdoubleSpinBox,"value");
  registerField("s2_theta",s2thetadoubleSpinBox,"value");
  registerField("s2_psi",s2psidoubleSpinBox,"value");
}

void SettingsPage::initializePage()
{
  bool env_fct = field("NGData").toBool();

  useRJMCMCcheckBox->setChecked(env_fct);
  useRJMCMCcheckBox->setEnabled(env_fct);

  s2alphalabel->setEnabled(env_fct);
  s2_alphadoubleSpinBox->setEnabled(env_fct);

  a_taulabel->setEnabled(env_fct);
  a_taudoubleSpinBox->setEnabled(env_fct);

  b_taulabel->setEnabled(env_fct);
  b_taudoubleSpinBox->setEnabled(env_fct);

  psilabel->setEnabled(!env_fct);
  psidoubleSpinBox->setEnabled(!env_fct);

  s2_psilabel->setEnabled(env_fct);
  s2psidoubleSpinBox->setEnabled(env_fct);
}

int SettingsPage::nextId() const
  {
    return Wizard::Page_Output;
  }

void SettingsPage::setEnabled()
{
  omegalabel->setEnabled(useFmodelcheckBox->isChecked());
  omegadoubleSpinBox->setEnabled(useFmodelcheckBox->isChecked());

  xilabel->setEnabled(useFmodelcheckBox->isChecked());
  xidoubleSpinBox->setEnabled(useFmodelcheckBox->isChecked());

  s2thetalabel->setEnabled(useFmodelcheckBox->isChecked());
  s2thetadoubleSpinBox->setEnabled(useFmodelcheckBox->isChecked());
}

OutputPage::OutputPage(QWidget *parent)
    : QWizardPage(parent)
{
  QVBoxLayout *vboxLayout = new QVBoxLayout();
  vboxLayout->setSpacing(6);
  vboxLayout->setMargin(9);
  vboxLayout->setObjectName(QString::fromUtf8("vboxLayout"));

  setTitle(tr("Ouput options"));

  RawMCMCgroupBox = new QGroupBox();
  RawMCMCgroupBox->setObjectName(QString::fromUtf8("RawMCMCgroupBox"));
  QVBoxLayout *vboxLayout1 = new QVBoxLayout(RawMCMCgroupBox);
  vboxLayout1->setSpacing(6);
  vboxLayout1->setMargin(9);
  vboxLayout1->setObjectName(QString::fromUtf8("vboxLayout1"));

  loglikcheckBox = new QCheckBox(RawMCMCgroupBox);
  loglikcheckBox->setObjectName(QString::fromUtf8("loglikcheckBox"));
  loglikcheckBox->setChecked(true);
  vboxLayout1->addWidget(loglikcheckBox);

  regcheckBox = new QCheckBox(RawMCMCgroupBox);
  regcheckBox->setObjectName(QString::fromUtf8("regcheckBox"));
  regcheckBox->setChecked(true);
  vboxLayout1->addWidget(regcheckBox);

  psicheckBox = new QCheckBox(RawMCMCgroupBox);
  psicheckBox->setObjectName(QString::fromUtf8("psicheckBox"));
  psicheckBox->setChecked(true);
  vboxLayout1->addWidget(psicheckBox);

  mcheckBox = new QCheckBox(RawMCMCgroupBox);
  mcheckBox->setObjectName(QString::fromUtf8("mcheckBox"));
  mcheckBox->setChecked(true);
  vboxLayout1->addWidget(mcheckBox);

  migcheckBox = new QCheckBox(RawMCMCgroupBox);
  migcheckBox->setObjectName(QString::fromUtf8("migcheckBox"));
  migcheckBox->setChecked(true);
  vboxLayout1->addWidget(migcheckBox);

  FcheckBox = new QCheckBox(RawMCMCgroupBox);
  FcheckBox->setObjectName(QString::fromUtf8("FcheckBox"));
  FcheckBox->setChecked(true);
  vboxLayout1->addWidget(FcheckBox);

  pcheckBox = new QCheckBox(RawMCMCgroupBox);
  pcheckBox->setObjectName(QString::fromUtf8("pcheckBox"));
  pcheckBox->setChecked(true);
  vboxLayout1->addWidget(pcheckBox);


  fstcheckBox = new QCheckBox(RawMCMCgroupBox);
  fstcheckBox->setObjectName(QString::fromUtf8("fstcheckBox"));
  fstcheckBox->setChecked(true);
  vboxLayout1->addWidget(fstcheckBox);

  ptildcheckBox = new QCheckBox(RawMCMCgroupBox);
  ptildcheckBox->setObjectName(QString::fromUtf8("ptildcheckBox"));
  ptildcheckBox->setChecked(true);
  vboxLayout1->addWidget(ptildcheckBox);


  vboxLayout->addWidget(RawMCMCgroupBox);

  PosteriorstatgroupBox = new QGroupBox();
  PosteriorstatgroupBox->setObjectName(QString::fromUtf8("PosteriorstatgroupBox"));
  QVBoxLayout *vboxLayout2 = new QVBoxLayout(PosteriorstatgroupBox);
  vboxLayout2->setSpacing(6);
  vboxLayout2->setMargin(9);
  vboxLayout2->setObjectName(QString::fromUtf8("vboxLayout2"));

  sdcheckBox = new QCheckBox(PosteriorstatgroupBox);
  sdcheckBox->setObjectName(QString::fromUtf8("sdcheckBox"));
  sdcheckBox->setChecked(true);
  vboxLayout2->addWidget(sdcheckBox);

  MtcheckBox = new QCheckBox(PosteriorstatgroupBox);
  MtcheckBox->setObjectName(QString::fromUtf8("MtcheckBox"));
  MtcheckBox->setChecked(true);
  vboxLayout2->addWidget(MtcheckBox);

  PopcheckBox = new QCheckBox(PosteriorstatgroupBox);
  PopcheckBox->setObjectName(QString::fromUtf8("PopcheckBox"));
  PopcheckBox->setChecked(true);
  vboxLayout2->addWidget(PopcheckBox);

  vboxLayout->addWidget(PosteriorstatgroupBox);

  RawMCMCgroupBox->setTitle(QApplication::translate("Dialog", "Raw MCMC output files", 0, QApplication::UnicodeUTF8));

  loglikcheckBox->setText(QApplication::translate("Dialog", "Log-likelihood (lglik)", 0, QApplication::UnicodeUTF8));
  regcheckBox->setText(QApplication::translate("Dialog", "Regression parameters (reg)", 0, QApplication::UnicodeUTF8));
  psicheckBox->setText(QApplication::translate("Dialog", "Psi (psi)", 0, QApplication::UnicodeUTF8));
  mcheckBox->setText(QApplication::translate("Dialog", "Proportion of migrant genes (mig)", 0, QApplication::UnicodeUTF8));
  migcheckBox->setText(QApplication::translate("Dialog", "Individual migration rates (mtild)", 0, QApplication::UnicodeUTF8));
  pcheckBox->setText(QApplication::translate("Dialog", "Population allele frequencies (popfrq)", 0, QApplication::UnicodeUTF8));
  FcheckBox->setText(QApplication::translate("Dialog", "Population specific inbreeding coefficients (F)", 0, QApplication::UnicodeUTF8));
  fstcheckBox->setText(QApplication::translate("Dialog", "Local population Fsts (Fst)", 0, QApplication::UnicodeUTF8));
  ptildcheckBox->setText(QApplication::translate("Dialog", "Metapopulation allele frequencies (ancfrq)", 0, QApplication::UnicodeUTF8));

  PosteriorstatgroupBox->setTitle(QApplication::translate("Dialog", "Posterior statistics", 0, QApplication::UnicodeUTF8));
  sdcheckBox->setText(QApplication::translate("Dialog", "Standard deviation for each monitored parameter", 0, QApplication::UnicodeUTF8));
  MtcheckBox->setText(QApplication::translate("Dialog", "Individual assignments", 0, QApplication::UnicodeUTF8));
  PopcheckBox->setText(QApplication::translate("Dialog", "Assignments over populations", 0, QApplication::UnicodeUTF8));

  setLayout(vboxLayout);

  registerField("printlglik",loglikcheckBox,"checked");
  registerField("printreg",regcheckBox,"checked");
  registerField("printpsi",psicheckBox,"checked");
  registerField("printm",mcheckBox,"checked");
  registerField("printmig",migcheckBox,"checked");
  registerField("printp",pcheckBox,"checked");
  registerField("printF",FcheckBox,"checked");
  registerField("printFst",fstcheckBox,"checked");
  registerField("printptild",ptildcheckBox,"checked");
  registerField("printsd",sdcheckBox,"checked");
  registerField("printMt",MtcheckBox,"checked");
  registerField("printPop",PopcheckBox,"checked");
}

void OutputPage::initializePage()
{
  bool Fmodel = field("Fst").toBool();
  bool env_fct = field("NGData").toBool();

  loglikcheckBox->setChecked(true);

  regcheckBox->setChecked(env_fct);
  regcheckBox->setEnabled(env_fct);

  psicheckBox->setChecked(false);
  psicheckBox->setEnabled(env_fct);

  mcheckBox->setChecked(true);
  migcheckBox->setChecked(false);

  pcheckBox->setChecked(false);
  FcheckBox->setChecked(true);

  fstcheckBox->setChecked(false);
  fstcheckBox->setEnabled(Fmodel);

  ptildcheckBox->setChecked(false);
  ptildcheckBox->setEnabled(Fmodel);

  sdcheckBox->setChecked(true);
  MtcheckBox->setChecked(true);
  PopcheckBox->setChecked(true);
}

int OutputPage::nextId() const
  {
    return Wizard::Page_StartRuns;
  }

FinishPage::FinishPage(QWidget *parent)
    : QWizardPage(parent)
{

  QVBoxLayout *vboxLayout = new QVBoxLayout();
  vboxLayout->setSpacing(6);
  vboxLayout->setMargin(9);
  vboxLayout->setObjectName(QString::fromUtf8("vboxLayout"));

  setTitle(tr("<center><b>Ready to start analyses</b></center>"));

  frame = new QFrame();
  frame->setObjectName(QString::fromUtf8("frame"));
  frame->setFrameShape(QFrame::StyledPanel);
  frame->setFrameShadow(QFrame::Raised);
  QVBoxLayout *vboxLayout1 = new QVBoxLayout(frame);
  vboxLayout1->setSpacing(6);
  vboxLayout1->setMargin(9);
  vboxLayout1->setObjectName(QString::fromUtf8("vboxLayout1"));

  label = new QLabel(frame);
  label->setObjectName(QString::fromUtf8("label"));
  vboxLayout1->addWidget(label);

  YesradioButton = new QRadioButton(frame);
  YesradioButton->setObjectName(QString::fromUtf8("YesradioButton"));
  YesradioButton->setChecked(true);
  vboxLayout1->addWidget(YesradioButton);

  NoradioButton = new QRadioButton(frame);
  NoradioButton->setObjectName(QString::fromUtf8("NoradioButton"));
  vboxLayout1->addWidget(NoradioButton);

  label->setText(QApplication::translate("Dialog", "Do you want to launch MCMC runs?", 0, QApplication::UnicodeUTF8));
  YesradioButton->setText(QApplication::translate("Dialog", "Yes", 0, QApplication::UnicodeUTF8));
  NoradioButton->setText(QApplication::translate("Dialog", "No, I need to change some options and I will run them later.", 0, QApplication::UnicodeUTF8));

  vboxLayout->addWidget(frame);
  setLayout(vboxLayout);
  
  registerField("Start",YesradioButton);
}

int FinishPage::nextId() const
  {
    return -1;
  }

OpenPage::OpenPage(QWidget *parent)
    : QWizardPage(parent)
{

  QVBoxLayout *vboxLayout = new QVBoxLayout();
  vboxLayout->setSpacing(6);
  vboxLayout->setMargin(9);
  vboxLayout->setObjectName(QString::fromUtf8("vboxLayout"));

  setTitle(tr("Load previous results"));

  frame = new QFrame();
  frame->setObjectName(QString::fromUtf8("frame"));
  frame->setFrameShape(QFrame::StyledPanel);
  frame->setFrameShadow(QFrame::Raised);
  QHBoxLayout *hboxLayout = new QHBoxLayout(frame);
  hboxLayout->setSpacing(6);
  hboxLayout->setMargin(9);
  hboxLayout->setObjectName(QString::fromUtf8("hboxLayout"));

  projectfilelabel = new QLabel(frame);
  projectfilelabel->setObjectName(QString::fromUtf8("projectfilelabel"));
  hboxLayout->addWidget(projectfilelabel);

  projectfilelineEdit = new QLineEdit(frame);
  projectfilelineEdit->setObjectName(QString::fromUtf8("projectfilelineEdit"));
  hboxLayout->addWidget(projectfilelineEdit);

  projectfiletoolButton = new QToolButton(frame);
  projectfiletoolButton->setObjectName(QString::fromUtf8("projectfiletoolButton"));
  hboxLayout->addWidget(projectfiletoolButton);

  projectfilelabel->setText(QApplication::translate("Dialog", "Project file:", 0, QApplication::UnicodeUTF8));
  projectfiletoolButton->setText(QApplication::translate("Dialog", "...", 0, QApplication::UnicodeUTF8));

  projectfilelineEdit->setText(QDir::homePath());

  completer = new QCompleter(this);
  completer->setModel(new QDirModel(completer));
  projectfilelineEdit->setCompleter(completer);
  vboxLayout->addWidget(frame);
  setLayout(vboxLayout);

  connect(projectfiletoolButton,SIGNAL(clicked(bool)),this,SLOT(getFilename()));
  connect(projectfilelineEdit,SIGNAL(textChanged(QString)),this, SIGNAL(completeChanged()));

  registerField("bpj_filename",projectfilelineEdit);
}

bool OpenPage::isComplete() const
  {
    QFileInfo info(projectfilelineEdit->text());
    return info.isFile();
  }

int OpenPage::nextId() const
  {
    return -1;
  }

void OpenPage::getFilename()
{
  QString fn = QFileDialog::getOpenFileName(this,tr("Open BIMr project file"),QDir::homePath(),tr("BIMr project Files (*.bpj);;All Files (*)"));
  if (!fn.isEmpty())
    projectfilelineEdit->setText(fn);
}
